###### plotfamd.R ######
### This file is part of FAMD. By downloading, copying or using it, you agree to the Legal Disclaimer.
###
### (c) Copyright by Philipp M. Schlter (henceforth 'the author') 2013.
###
### *** Legal Disclaimer ***
###     FAMD is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY, 
###     whether expressed or implied; without even the implied warranty of MERCHANTABILITY 
###     or FITNESS FOR A PARTICULAR PURPOSE. The author will not be liable for any special, 
###     incidental, consequential, indirect or similar damages due to loss of data or any
###     other reason, even if the author or an agent of his has been advised of the 
###     possibility of such damages. In no event shall the author be liable for any damages,
###     regardless of the form of the claim. The person using the software bears all
###     risk as to the quality and performance of the software.
###
### *** Licence terms for use of source code ***
###     Licence terms: Artistic License 2.0                                  
###     A copy of the licence terms is available at: http://www.opensource.org/licenses/artistic-license-2.0.php          
###
### *** Comments ***
###     Essentially, in non-legal terms, the main idea of the terms of use are:
###     (1) use at your own risk
###     (2) this is free and open-source, but if you make any [significant] changes to the 
###         source code (particularly, bug-fixes or improvements), I would highly 
###         appreciate if you can let me have the changes, so that I can incorporate them 
###         in the distributed version for the benefit of all users.
###
### *** Do not change the following line (to enable FAMD to check the version) ***
RPLOTFAMDVERSION = "0.01" # Format: [Number].[TwoDigitNumber]

### *** Purpose of the file ***
###     To enable automated plotting of results recorded in FAMD's analysis output file.
###
### *** Interface with FAMD ***
###     This file will be called by FAMD via R's Rscript.exe like so:
###     RScript.exe ThisScript.R inputName pdfName [option_flags] [...]
###     where...
###          inputName: 	Name of the analysis results file to process.
###          pdfName: 		Name of PDF file in which graphical output of the script will be collected.
###		     option_flags:	Special optional flags for the script's input, in '/FLAG:[...]' format.
###	Note that inputName is expected to be argument 1 (arg[1]) and pdfName to be argument 2 (arg[2])
###
### *** Notes ***
###     Development of this script is on-going, it isn't very polished yet. You may wish to check
###     for newer versions from time to time. Also, feel free to contribute.



### START OF SCRIPT

# *** Obtaining command-line arguments
args <- commandArgs(trailingOnly = TRUE)	# get command-line arguments of script
nArgs <- length(args)

# *** Hard-coded Settings (do not change without changing FAMD source)
npInputName <- 1			# Number of command-line parameter specifying input file (FAMD analysis file)
npPdfName 	<- 2 			# Number of command-line parameter specifying pdf output file
npExpectedArg <- npPdfName 	# Number of file arguments expected before optional arguments can occur
# Constants to match FAMD output
cSpace 				<- " "
cTab 				<- "\t"
cOpenBracket 		<- "("
cCloseBracket 		<-")"
cMissingDataIndiv 	<- "Missing data in individuals"
cMissingDataLoc 	<- "Missing data in loci"
cFreqInd 			<- "Frequencies (band presences) in individuals"
cFreqLoc 			<- "Frequencies (band presences) in loci"
cNullAlleles 		<- "Null allele frequencies" # " for group AllData"
cCalcMeth			<- "Calculation method:" # " Bayesian (uniform prior)"
cHybIndex 			<-"Maximum Likelihood Hybrid Index Results"
cPopRealloc 		<- "Population reallocation results"
cAMOVA 				<- "AMOVA results"
# Constants for interpreting FAMD output
cSectionTypes 	<- c("MissingDataIndiv","MissingDataLoc","FreqInd","FreqLoc","NullAlleles","HybIndex","PopRealloc","AMOVA")
cList 			<- c(cMissingDataIndiv,cMissingDataLoc,cFreqInd,cFreqLoc,cNullAlleles,cHybIndex,cPopRealloc,cAMOVA)	# needs to match cSectionTypes layout
cOfs  			<- c(1,                1,              1,       1,       1,           2,        1,          0     )	# needs to match cSectionTypes layout
cBlnkIgnore 	<- c(0,                0,              0,       0,       0,           0,        0,          3     )	# needs to match cSectionTypes layout
# Constants to match optional FLAGS argument
cFlags 				<- "/FLAGS:"
cFlagLastOnly 		<- "L"

# *** Settings and Parameters (may be changed by user)
defaultinputname <- "analysis.txt"	# Default name of FAMD's analysis output file
defaultpdfname <- "famdplot.pdf"	# Default name to write graphs to
currpath <- "" 						# Set path name here if you want to use the script from the R GUI
ToPDF <- TRUE 						# Shall output be written to PDF file (rather than display)?	
DoSetPath <- FALSE					# Shall the script explicitly set the path?
flagAddCI2 <- FALSE					# Add CI at alpha2 level to (selected) graphs
flagProcessLastSectionOnly <- FALSE	# Only process last section in file
# Default alpha (significance) levels for condifence intervals
alpha1 <- 0.05 	# 0.05 -> 95% CI, etc.
alpha2 <- 0.01 
#alpha3 <- 0.001 # currently not used!

# Colour settings
ForeCol 	<- "red"	# Foreground colours 1,2,3
ForeCol2 	<- "blue"
ForeCol3 	<- "grey"
nDefCol 	<- 36		# Number of default colours to be made, e.g. for heat maps
MakeRefCol <- function (nCol) {return(rainbow(nCol))}			# Function to make reference colours
MakeHeatMapCol <- function (nCol) {return(heat.colors(nCol))} 	# Function to make heatmap colours
RevHeatMapColours <- MakeHeatMapCol(nDefCol)[nDefCol:1]			# Default reverse-order heatmap colours

# *** Starting to process input
# Evaluating expected command-line arguments
if (nArgs > (npInputName-1)) {inputfile <- args[npInputName]} else {inputfile <- defaultinputname}
if (nArgs > (npPdfName-1)) {pdfname <- args[npPdfName]} else {pdfname <- defaultpdfname}
# Getting optional FLAGS argument, if available
if (nArgs > npExpectedArg) {
	argFlags <- args[npExpectedArg + 1]	
	if (length(grep (cFlags, argFlags))>0) {
		argFlags <- sub (cFlags, "", argFlags)
		flagProcessLastSectionOnly <- (length(grep(cFlagLastOnly, argFlags))>0)
	}
}
# Calculating values for confidence intervals at different alpha levels
z1 <- qnorm(1-alpha1/2) # cdf= cumulative normal distribution function, z1 ~1.96 for alpha=0.05
z2 <- qnorm(1-alpha2/2)
#z3 <- qnorm(1-alpha3/2) # currently not used!

### FUNCTIONS
# Notes: Two functions (for each type of data section) dermine what is plotted: 
#        a) a function that parses input data
#        b) a function that plotted the parsed data
#	     Obviously, if function (a) is changed, you may need to adapt function (b) so that it understands the output of (a)

# *** Functions to parse input file data
# * Function to get section start lines
#   - rawdata: raw input file data lines
GetStartLines <- function (rawdata) {
	nL <- length(cList)
	StartLine <- NULL
	HitType <- NULL
	for (i in (1:nL)) {
		currHits <- grep(cList[i],rawdata,fixed=T)
		StartLine <- c(StartLine, currHits)
		nH <- length(currHits)
		if (nH > 0) { HitType <- c (HitType, rep(i,nH))}
	}
	return (rbind(StartLine,HitType))
}

# * Function to get sections (WITH header lines!)
#   - rawdata: raw input file data lines
GetSections <- function (rawdata) {
	StartLineData <- GetStartLines (rawdata)	
	#StartLines <- rawdata[StartLineData["StartLine",]]
	nStarts <- dim(StartLineData)[2]
	Sections <- matrix (nrow = 3, ncol = nStarts)
	rownames(Sections)<-c("Start","Stop","Type")
	lEmpty <- which(rawfile == "")
	for (i in 1:nStarts) {
		lStart <- StartLineData["StartLine",i]		
		Hit <- StartLineData["HitType",i]		
		nEmptyIgnore <-  cBlnkIgnore[Hit]
		lStartX <- lStart + cOfs[Hit]+1
		lEmptyOK <- lEmpty[which (lEmpty>= lStartX)]
		lEmptyOK <- lEmptyOK[(nEmptyIgnore+1):length(lEmptyOK)]
		lStop <- min(lEmptyOK[lEmptyOK > lStartX])-1
		Sections[1,i] <- lStart
		Sections[2,i] <- lStop
		Sections[3,i] <- Hit
	}
	return(Sections)
}

# * Function to extract text from within brackets
#   - Str: character string from which to extract text
ExtractFromBrackets <- function (Str) {return(gsub (cCloseBracket,"",strsplit (Str,split=c(cOpenBracket,cCloseBracket),fixed=TRUE)[[1]][2]))}

# * Function to extract a single section of raw file data to a table
#   - rawdata: raw input file data lines
#   - lStart: start line of section in rawdata
#   - lStop: stop line of section in rawdata
#   - delimChar: delimiter char that separates data elements
SectionToTableEx <- function (rawdata, lStart, lStop, delimChar) {
	section <- rawdata[lStart:lStop]
	res <- NULL
	for (i in (lStart:lStop-lStart+1)) {
		parsedline <- unlist(strsplit (section[i], delimChar))
		parsedline <- parsedline[parsedline != ""]
		res <- rbind(res, parsedline)
	}
	rownames (res) <- 1:dim(res)[1]
	return (res)
}

# * Function to extract a single section of raw file data to a table (as above), with a Space character as delimiter
SectionToTableSpc <- function (rawdata, lStart, lStop) {return(SectionToTableEx(rawdata, lStart, lStop, cSpace))}

# * Function to extract a single section of raw file data to a table (as above), with a Tab character as delimiter
SectionToTableTab <- function (rawdata, lStart, lStop) {return(SectionToTableEx(rawdata, lStart, lStop, cTab))}

# * Function to extract a single "hybrid index" section of raw file data to a table / list
#   - rawdata: raw input file data lines
#   - lStart: start line of section in rawdata
#   - lStop: stop line of section in rawdata
SectionToTableTabHI <- function (rawdata, lStart, lStop) {
	# HI-section specific constants
	chiExtraHdrLines <- 2
	# code
	S <- NULL
	S$Section <- SectionToTableEx(rawdata, lStart, lStop,cTab)
	Hdr <- NULL
	for (i in (1:chiExtraHdrLines)) {
		Hdr <- c(Hdr, rawdata[lStart-i])
	}
	S$Hdr <- Hdr
	return(S)
}

# * Function to extract a single "AMOVA" section of raw file data to a table / list
#   - rawdata: raw input file data lines
#   - lStart: start line of section in rawdata
#   - lStop: stop line of section in rawdata
SectionToTableAmova <- function (rawdata, lStart, lStop) {
	# AMOVA-section-specific constants
	cavLCoeff <- 1
	cavPhiST <- "PhiST"
	cavPhiSR <- "PhiSR"
	cavPhiRT <- "PhiRT"
	cavVarSubSect <- 2
	cavPctVarCol <- 4
	# code
	section <- rawdata[lStart:lStop]
	S<-NULL
	S$CoeffInfo <- section[cavLCoeff]
	S$PhiST <- as.numeric(unlist(strsplit(section[grep (cavPhiST, section)],cSpace))[2])
	S$PhiSR <- as.numeric(unlist(strsplit(section[grep (cavPhiSR, section)],cSpace))[2])
	S$PhiRT <- as.numeric(unlist(strsplit(section[grep (cavPhiRT, section)],cSpace))[2])
	emptyLines <- which(section == "")
	VarSect <- section[(emptyLines[cavVarSubSect]+1) : (emptyLines[cavVarSubSect+1]-1)]
	nVarLines <- length(VarSect)
	VarComponents <- NULL
	for (i in 1:(nVarLines-1)) {VarComponents  <- c(VarComponents ,ExtractFromBrackets (VarSect[i]))}
	VarSect <-gsub ("/Within ","",VarSect) # just remove one space from nested AMOVA result to aid parsing
	S$PctVariance <- as.numeric(SectionToTableSpc (VarSect,1,nVarLines)[,cavPctVarCol][1:(nVarLines-1)])
	names(S$PctVariance) <- VarComponents
	return(S)
}

# *** Functions for plotting of data
# * Helper function to add vertical (Y) lines to a plot
#   - XPos: x position at which to place Y lines in plot
#   - YMin: lower Y position at which lines for each position will start (has to match XPos)
#   - YMax: upper Y position at which lines for each position will end (has to match XPos)
#   - ...: further graphical arguments to be passed to "lines"
YLines <- function (XPos, YMin, YMax, ...) {
	L <- length(XPos)
	if (L==0) {stop("No X positions")}
	if (length(YMin) != L) {stop("XPos and YMin do not match")}
	if (length(YMax) != L) {stop("XPos and YMin do not match")}
	for (i in (1:L)) {
		lines (c(XPos[i],XPos[i]),c(YMin[i],YMax[i]),...)
	}
}

# * Function to plot data from Missing Data section
#   - section: input section as table
#   - indiv: if TRUE, items to be plotted are assumed to be individuals, loci otherwise
PlotMDStats <- function (section, indiv=FALSE){
	if (indiv == TRUE) {
		MainTitle <- "Missing data in individuals"
		XLabel <- "Individuals"
	} else {
		XLabel <- "Loci"
		MainTitle <- "Missing data in loci"
	}
	par (mfrow = c(2,1))
	MD <- as.double(section[,2])
	meanMD <- mean(MD)
	sdMD <- sd(MD)
	n <- length(MD)
	seMD <- sdMD / sqrt(n)
	ciDiff1 <- z1 * seMD
	ciLow1 <- meanMD - ciDiff1
	ciHigh1 <- meanMD + ciDiff1
	if (flagAddCI2) {
		ciDiff2 <- z2 * seMD
		ciLow2 <- meanMD - ciDiff2
		ciHigh2 <- meanMD + ciDiff2
	}

	MaxVal <- max(MD)*1.2
	barplot(names.arg=section [,1],MD,ylim =c(0,MaxVal), col = ForeCol,main = MainTitle, ylab="Percentage missing data",xlab=XLabel, las=2)
	abline (h=meanMD, col = ForeCol2)
	abline (h=ciLow1, col = ForeCol2, lty= 2)
	abline (h=ciHigh1, col = ForeCol2, lty= 2)
	if (flagAddCI2) {
		abline (h=ciLow2, col = ForeCol2, lty= 3)
		abline (h=ciHigh2, col = ForeCol2, lty= 3)
	}

	hist (MD, breaks = 25, col = ForeCol,xlim = c(0,max(30,MD)),xlab="Percentage of missing data",main=paste(MainTitle,"as histogram",sep=" "))
	abline (v=meanMD, col= ForeCol2)
	abline (v=ciLow1, col = ForeCol2, lty= 2)
	abline (v=ciHigh1, col = ForeCol2, lty= 2)
	if (flagAddCI2) {
		abline (v=ciLow2, col = ForeCol2, lty= 3)
		abline (v=ciHigh2, col = ForeCol2, lty= 3)
	}
}

# * Function to plot data from Individuals in Missing Data section
#   - X: input section as table
PlotMDIndStats <- function (X) {return(PlotMDStats(X,indiv=T))}

# * Function to plot data from Loci in Missing Data section
#   - X: input section as table
PlotMDLocStats <- function (X) {return(PlotMDStats(X,indiv=F))}

# * PLACEHOLDER Function to plot band frequency data --- NOT YET IMPLEMENTED
#   - section: input section as table
#   - indiv: if TRUE, items to be plotted are assumed to be individuals, loci otherwise
PlotFreqStats <- function (section, indiv=FALSE){
	print ("PlotFreqStats not yet implemented")
}

# * PLACEHOLDER Function to plot individual band frequency data --- NOT YET IMPLEMENTED
#   - X: input section as table
PlotFreqInd <- function (X) {return(PlotFreqStats (X,indiv=T))}

# * PLACEHOLDER Function to plot locus band frequency data --- NOT YET IMPLEMENTED
#   - X: input section as table
PlotFreqLoc <- function (X) {return(PlotFreqStats (X,indiv=F))}

# * PLACEHOLDER Function to plot allele frequency data --- NOT YET IMPLEMENTED
#   - section: input section as table
PlotAlleles  <- function (section){
	print ("PlotAlleles  not yet implemented")
}

# * Function to plot hybrid indices
#   - data: input section data as list
PlotHIndex <- function (data) {
	# HI Constants
	chiErrCol1 <- 6
	chiNErrCol <- 2
	hiPlotLikelihoods <- FALSE

	# Annotate and parse input data
	section <- data$Section
	hdr <- data$Hdr
	colnames(section) <- section[1,]
	rownames(section) <- section[,1]	
	hiErr <- section[-1,c("ErrorH","ErrorLB","ErrorUB")]
	hiCol <- c("h","LowerBound","UpperBound")
	hi <- matrix(as.numeric(section[-1,hiCol]),ncol=3,dimnames=list(rownames(section[-1,]),hiCol))
	hiOrd <- order(hi[,"h"], decreasing = T)
	hi <- hi[hiOrd,]
	LnL <- section[-1,"lnL"]
	LnL <- LnL[hiOrd]
	LnL <- matrix(as.numeric(LnL),dimnames=list(names(LnL),c("LnL")))
	# Parsing info on parents at h=0 and h=1
	parent <- NULL
	for (i in (0:1)) {
		ptmp <- hdr[grep(paste("h=",i,sep=""),hdr)]
		ptmp2 <- strsplit(ptmp," ")
		parent[i+1] <- ptmp2[[1]][length(ptmp2[[1]])]
		parent[i+1] <- gsub ("AllData/","",parent[i+1])
	}

	# Plotting data
	if (hiPlotLikelihoods) {par (mfrow=c(2,1))} else {par (mfrow=c(1,1))}
	# Plotting Hyb Index values
	plot(hi[,"h"],type="b",pch=19,col=ForeCol,ylim=c(0,1),xlab="",main="Maximum-likelihood hybrid indices",ylab="Hybrid Index",axes=FALSE)
	YLines (1:nrow(hi),hi[,"LowerBound"],hi[,"UpperBound"],lty=1,lwd=0.5,col=ForeCol3)
	abline(h= c(0,0.5,1),lty=c(3,2,3), col=ForeCol2)
	axis (2)
	axis (1, at = (1:nrow(hi)), labels= rownames(hi),las=2)
	axis (4, at = c(0,0.5,1), labels=c(parent[1],"F1",parent[2]))
	box()
	# Plotting Likelihoods
	if (hiPlotLikelihoods) {
		barplot(-LnL[,1],horiz=F,names.arg=rownames(LnL),las=2,col=ForeCol,ylab="-ln Likelihood",main="Likelihoods for hybrid indices")
	}
}

# * Function to plot Population Reallocation data
#   - section: input section data as list
PlotPopRealloc <- function (section){
	#Section-specific constants
	cprGrpH0 <- "Group_H0:"
	cprLLD <- "Log-Likelihood Difference:"
	cprAlloc <- "Allocation:"
	nprExtraRows <- 4

	PopReallocSection <- section
	# begin plotting of PopReallocSection
	colnames (PopReallocSection) <- PopReallocSection [1,]
	rownames (PopReallocSection) <- PopReallocSection [,1]
	nGrpTested <- nrow(PopReallocSection) - nprExtraRows 
	pr <- t(PopReallocSection [-1,-1])

	# Setting graphics parameters
	par (mfrow = c(1,1))

	# Likelihoods
	prLike<- matrix(as.numeric(pr [,-(nGrpTested+1:ncol(PopReallocSection))]),nrow=nrow(pr),dimnames=list(rownames (pr),colnames (pr)[1:nGrpTested]))
	# Log-likelihood differences best vs 2ndbest
	prLLD <- as.numeric(pr[,cprLLD])
	names(prLLD) <- rownames(pr)
	# Null-hypothesis vs assignment for individuals
	prTmp1 <- as.factor(c(pr[,c(cprGrpH0,cprAlloc)]))
	levels (prTmp1) <- c(levels (prTmp1), "None") 	
	prTmp1 <- relevel (prTmp1, ref="None")
	AllocLevels <- levels(prTmp1)
	AllocTargetLevels <- levels(as.factor(colnames(prLike)))
	PossibleAllocResults <- AllocLevels %in% AllocTargetLevels
	PossibleAllocResults[1] <- TRUE 	# It is always possible that allocation goes to "None"
	# Allocations
	prAlloc <- cbind(prTmp1[1:nrow(pr)],prTmp1[(nrow(pr)+1):length(prTmp1)])
	colnames(prAlloc) <- c("H0","Alloc")
	rownames(prAlloc) <- rownames (pr)

	# Plotting Null hypothesis and ML Allocation
	RefCol <- c("grey", MakeRefCol (length(AllocLevels)-1))	
	image(x=c(0,1,2)/3,y=1:nrow(prAlloc),z=t(prAlloc), col=RefCol,zlim = c(1,length(RefCol)),axes=F,xlim=c(0,1),xlab="",ylab="",main="Hypothesised vs ML allocated group membership")
	axis(2, at=(1:nrow(prAlloc)),labels= rownames(prAlloc),las=2,tick=F)
	axis(1, at=c(0,1)/3+1/6,labels=colnames(prAlloc),tick=F)
	legend ("right",legend=AllocLevels, fill = RefCol[1:length(AllocLevels)],title="Group",border=c("white","black")[as.numeric(PossibleAllocResults)+1] )

	#Plotting LLDs
	image (t(as.matrix(prLLD)),col=RevHeatMapColours,axes=F, main = "Log-likelihood differences for allocation")
	image(as.matrix(1:length(RevHeatMapColours)),col=RevHeatMapColours,axes=F,main="Heatmap colour scale",xlab= "Min to Max",add=T)
	axis (1,at=c(-0.5,0.5),labels=c("Log-Likelihood difference","Heat colour scale (Min to Max)"),tick=F)	
	axis (2,at=(1:nrow(prAlloc))/nrow(prAlloc),labels= rownames(prAlloc),las=2, tick=F,cex=2)
	
	# Heatmap of assignment likelihoods
	AllocLevID <- which (AllocLevels %in% colnames(prLike))
	
	#heatmap(prLike[,AllocLevels[AllocLevID]],Rowv=NA,Colv=NA,RowSideColors = RefCol[as.integer(prAlloc$H0[rownames(prLike)])],ColSideColors = RefCol[AllocLevID],col=RevHeatMapColours )
	par (mfrow=c(1,1))
	heatmap(prLike[,AllocLevels[AllocLevID]],Rowv=NA,Colv=NA,RowSideColors = RefCol[prAlloc[rownames(prLike),"H0"]],ColSideColors = RefCol[AllocLevID],col=RevHeatMapColours,main="Heatmap of allocation likelihoods" )
}

# * Function to plot AMOVA data
#   - section: input section data as list
PlotAmova <- function (section) {
	# currently only plot PctVariance (% Variance) components
	pie(section$PctVariance, col=MakeRefCol(3)[length(section$PctVariance):1], main = "Partitioning of AMOVA variance components", sub=section$CoeffInfo)
}


### SCRIPT MAIN 

# Setting path if required
if (nArgs == 0) {if(DoSetPath){setwd(currpath)}}	#

# Mapping of section types to plotting functions
Section2TableFunctions   <- c(SectionToTableSpc, SectionToTableSpc, SectionToTableSpc, SectionToTableSpc, SectionToTableSpc, SectionToTableTabHI, SectionToTableTab, SectionToTableAmova) #needs to match cSectionTypes layout
SectionPlottingFunctions <- c(PlotMDIndStats , PlotMDLocStats, PlotFreqInd, PlotFreqLoc, PlotAlleles, PlotHIndex, PlotPopRealloc, PlotAmova ) #needs to match cSectionTypes layout

# Writing out version info
print(paste("FAMD R Plotting Script Version",RPLOTFAMDVERSION))

# *** Processing input data 
# Reading data
rawfile <- readLines (inputfile)	# read input file line-by-line
nSect 	<- 0

# Getting sections
if (length(rawfile[GetStartLines (rawfile)["StartLine",]])>0) {
	sect <- GetSections(rawfile)
	nSect <- ncol(sect)
	sect <- sect[,order(sect["Start",])]
}
if (nSect == 1) {sect <- as.matrix(sect)}

# Processing and plotting sections
if (nSect >0) {		# if there are valid sections in the input file...
	SelectedSection <- nSect
	if (flagProcessLastSectionOnly == FALSE) {
		# in this case, we want to plot all sections...
		SelectedSection <- 1:nSect
	} 
	if (ToPDF) {pdf(pdfname)}	# Starting PDF output
	for (s in SelectedSection) {
		# Processing input sections
		stype <-sect["Type",s]
		CurrSect <- NULL
		print(paste("Processing a",cSectionTypes[stype] ,"section: lines",sect["Start",s],"-",sect["Stop",s]))
		CurrSect <- Section2TableFunctions[[stype]](rawfile,sect["Start",s]+1+cOfs[sect["Type",s]],sect["Stop",s])
		SectionPlottingFunctions[[stype]](CurrSect) 
	}
	if (ToPDF) {dev.off()}		# Stopping PDF output
} else {
	print("ERROR: could not find valid input!")
}

### END OF SCRIPT